<?php

/**
 * ZITEO Authentication Handler
 */

require_once __DIR__ . '/db.php';
require_once __DIR__ . '/functions.php';

/**
 * Register a new user
 */
function registerUser(string $name, string $email, string $password, string $phone): array
{
    $pdo = db();

    // Check if email exists
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$email]);
    if ($stmt->fetch()) {
        return ['success' => false, 'error' => 'Email already registered'];
    }

    // Hash password
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);

    // Insert user
    $stmt = $pdo->prepare("
        INSERT INTO users (name, email, password, phone, status, created_at) 
        VALUES (?, ?, ?, ?, 'active', NOW())
    ");
    $stmt->execute([$name, $email, $hashedPassword, $phone]);

    $userId = $pdo->lastInsertId();

    // Start session
    startUserSession($userId);

    return ['success' => true, 'user_id' => $userId];
}

/**
 * Login user
 */
function loginUser(string $email, string $password): array
{
    $pdo = db();

    $stmt = $pdo->prepare("SELECT id, password, status FROM users WHERE email = ?");
    $stmt->execute([$email]);
    $user = $stmt->fetch();

    if (!$user) {
        return ['success' => false, 'error' => 'Invalid email or password'];
    }

    if (!password_verify($password, $user['password'])) {
        return ['success' => false, 'error' => 'Invalid email or password'];
    }

    if ($user['status'] === 'suspended') {
        return ['success' => false, 'error' => 'Your account has been suspended'];
    }

    // Start session
    startUserSession($user['id']);

    return ['success' => true, 'user_id' => $user['id']];
}

/**
 * Start user session
 */
function startUserSession(int $userId): void
{
    session_regenerate_id(true);
    $_SESSION['user_id'] = $userId;
    $_SESSION['logged_in_at'] = time();

    // Load user's store if exists
    $pdo = db();
    $stmt = $pdo->prepare("SELECT id FROM stores WHERE user_id = ? LIMIT 1");
    $stmt->execute([$userId]);
    $store = $stmt->fetch();

    if ($store) {
        $_SESSION['store_id'] = $store['id'];
    }
}

/**
 * Logout user
 */
function logoutUser(): void
{
    $_SESSION = [];

    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(
            session_name(),
            '',
            time() - 42000,
            $params["path"],
            $params["domain"],
            $params["secure"],
            $params["httponly"]
        );
    }

    session_destroy();
}

/**
 * Get current user data
 */
function getCurrentUser(): ?array
{
    if (!isLoggedIn()) {
        return null;
    }

    $pdo = db();
    $stmt = $pdo->prepare("SELECT id, name, email, phone, status, created_at FROM users WHERE id = ?");
    $stmt->execute([currentUserId()]);
    return $stmt->fetch() ?: null;
}

/**
 * Admin login
 */
function loginAdmin(string $email, string $password): array
{
    $pdo = db();

    $stmt = $pdo->prepare("SELECT id, password, role FROM admin_users WHERE email = ?");
    $stmt->execute([$email]);
    $admin = $stmt->fetch();

    if (!$admin || !password_verify($password, $admin['password'])) {
        return ['success' => false, 'error' => 'Invalid credentials'];
    }

    session_regenerate_id(true);
    $_SESSION['admin_id'] = $admin['id'];
    $_SESSION['admin_role'] = $admin['role'];

    return ['success' => true];
}

/**
 * Check if admin is logged in
 */
function isAdminLoggedIn(): bool
{
    return isset($_SESSION['admin_id']);
}

/**
 * Require admin authentication
 */
function requireAdmin(): void
{
    if (!isAdminLoggedIn()) {
        redirect('/admin/login.php');
    }
}
