<?php

/**
 * Products API Endpoints
 */

function getProducts(): void
{
    $storeId = $_GET['store_id'] ?? null;

    if (!$storeId) {
        jsonError('store_id is required');
    }

    $pdo = db();
    $stmt = $pdo->prepare("
        SELECT id, name, description, price, offer_price, image, stock_status
        FROM products
        WHERE store_id = ? AND is_active = 1
        ORDER BY created_at DESC
    ");
    $stmt->execute([$storeId]);
    $products = $stmt->fetchAll();

    jsonSuccess($products);
}

function getProduct(int $id): void
{
    $pdo = db();
    $stmt = $pdo->prepare("
        SELECT p.*, s.name as store_name, s.whatsapp
        FROM products p
        JOIN stores s ON p.store_id = s.id
        WHERE p.id = ? AND p.is_active = 1
    ");
    $stmt->execute([$id]);
    $product = $stmt->fetch();

    if (!$product) {
        jsonError('Product not found', 404);
    }

    jsonSuccess($product);
}

function createProduct(): void
{
    requireAuth();
    $storeId = currentStoreId();

    if (!$storeId) {
        jsonError('No store found. Create a store first.');
    }

    // Check product limit
    $pdo = db();
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as count FROM products WHERE store_id = ?
    ");
    $stmt->execute([$storeId]);
    $count = $stmt->fetch()['count'];

    $stmt = $pdo->prepare("
        SELECT p.product_limit FROM plans p
        JOIN stores s ON s.plan_id = p.id
        WHERE s.id = ?
    ");
    $stmt->execute([$storeId]);
    $limit = $stmt->fetch()['product_limit'] ?? 15;

    if ($count >= $limit) {
        jsonError("Product limit reached ($limit). Upgrade your plan.");
    }

    // Handle multipart form data or JSON
    $name = $_POST['name'] ?? '';
    $description = $_POST['description'] ?? '';
    $price = (int)($_POST['price'] ?? 0) * 100; // Convert to paise
    $offerPrice = !empty($_POST['offer_price']) ? (int)$_POST['offer_price'] * 100 : null;

    if (empty($name)) {
        jsonError('Product name is required');
    }

    if ($price <= 0) {
        jsonError('Valid price is required');
    }

    // Handle image upload
    $imagePath = null;
    if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = UPLOAD_PATH . '/' . $storeId;
        $imagePath = handleUpload($_FILES['image'], $uploadDir);
    }

    $stmt = $pdo->prepare("
        INSERT INTO products (store_id, name, description, price, offer_price, image, stock_status, is_active, created_at)
        VALUES (?, ?, ?, ?, ?, ?, 'in_stock', 1, NOW())
    ");
    $stmt->execute([$storeId, $name, $description, $price, $offerPrice, $imagePath]);

    jsonSuccess(['product_id' => $pdo->lastInsertId()], 'Product created successfully');
}

function updateProduct(int $id): void
{
    requireAuth();
    $storeId = currentStoreId();

    $pdo = db();

    // Verify ownership
    $stmt = $pdo->prepare("SELECT id FROM products WHERE id = ? AND store_id = ?");
    $stmt->execute([$id, $storeId]);
    if (!$stmt->fetch()) {
        jsonError('Product not found', 404);
    }

    $data = json_decode(file_get_contents('php://input'), true);

    $updates = [];
    $params = [];

    if (isset($data['name'])) {
        $updates[] = 'name = ?';
        $params[] = $data['name'];
    }
    if (isset($data['description'])) {
        $updates[] = 'description = ?';
        $params[] = $data['description'];
    }
    if (isset($data['price'])) {
        $updates[] = 'price = ?';
        $params[] = (int)$data['price'] * 100;
    }
    if (isset($data['offer_price'])) {
        $updates[] = 'offer_price = ?';
        $params[] = $data['offer_price'] ? (int)$data['offer_price'] * 100 : null;
    }
    if (isset($data['stock_status'])) {
        $updates[] = 'stock_status = ?';
        $params[] = $data['stock_status'];
    }
    if (isset($data['is_active'])) {
        $updates[] = 'is_active = ?';
        $params[] = $data['is_active'] ? 1 : 0;
    }

    if (empty($updates)) {
        jsonError('No fields to update');
    }

    $params[] = $id;
    $stmt = $pdo->prepare("UPDATE products SET " . implode(', ', $updates) . " WHERE id = ?");
    $stmt->execute($params);

    jsonSuccess([], 'Product updated');
}

function deleteProduct(int $id): void
{
    requireAuth();
    $storeId = currentStoreId();

    $pdo = db();
    $stmt = $pdo->prepare("DELETE FROM products WHERE id = ? AND store_id = ?");
    $stmt->execute([$id, $storeId]);

    if ($stmt->rowCount() === 0) {
        jsonError('Product not found', 404);
    }

    jsonSuccess([], 'Product deleted');
}
