<?php

/**
 * Stores API Endpoints
 */

function getStoreBySlug(): void
{
    $slug = $_GET['slug'] ?? '';

    if (empty($slug)) {
        jsonError('Store slug is required');
    }

    $pdo = db();
    $stmt = $pdo->prepare("
        SELECT s.id, s.name, s.slug, s.whatsapp, s.status, s.theme_config,
               u.name as owner_name
        FROM stores s
        JOIN users u ON s.user_id = u.id
        WHERE s.slug = ? AND s.status = 'active'
    ");
    $stmt->execute([$slug]);
    $store = $stmt->fetch();

    if (!$store) {
        jsonError('Store not found', 404);
    }

    // Parse theme config
    $store['theme_config'] = json_decode($store['theme_config'], true) ?: [];

    jsonSuccess($store);
}

function createStore(): void
{
    requireAuth();

    $data = json_decode(file_get_contents('php://input'), true);

    $name = trim($data['name'] ?? '');
    $slug = slugify($data['slug'] ?? $name);
    $whatsapp = trim($data['whatsapp'] ?? '');

    // Validation
    if (empty($name) || strlen($name) < 3) {
        jsonError('Store name is required (min 3 characters)');
    }

    if (empty($whatsapp) || !isValidPhone($whatsapp)) {
        jsonError('Valid WhatsApp number is required');
    }

    $pdo = db();

    // Check if user already has a store
    $stmt = $pdo->prepare("SELECT id FROM stores WHERE user_id = ?");
    $stmt->execute([currentUserId()]);
    if ($stmt->fetch()) {
        jsonError('You already have a store');
    }

    // Check slug uniqueness
    $stmt = $pdo->prepare("SELECT id FROM stores WHERE slug = ?");
    $stmt->execute([$slug]);
    if ($stmt->fetch()) {
        $slug = $slug . '-' . rand(100, 999);
    }

    // Create store
    $stmt = $pdo->prepare("
        INSERT INTO stores (user_id, plan_id, name, slug, whatsapp, status, theme_config, created_at)
        VALUES (?, 1, ?, ?, ?, 'pending', '{}', NOW())
    ");
    $stmt->execute([currentUserId(), $name, $slug, $whatsapp]);

    $storeId = $pdo->lastInsertId();
    $_SESSION['store_id'] = $storeId;

    jsonSuccess([
        'store_id' => $storeId,
        'slug' => $slug,
        'status' => 'pending'
    ], 'Store created! Awaiting admin approval.');
}
