<?php

/**
 * Products Management Page
 */
session_start();
require_once dirname(__DIR__, 2) . '/includes/auth.php';
requireAuth();

$storeId = currentStoreId();
if (!$storeId) {
    header('Location: create-store.php');
    exit;
}

$pdo = db();
$action = $_GET['action'] ?? 'list';
$error = '';
$success = flash('success');

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $price = (int)($_POST['price'] ?? 0);
    $offerPrice = !empty($_POST['offer_price']) ? (int)$_POST['offer_price'] : null;
    $stockStatus = $_POST['stock_status'] ?? 'in_stock';

    if (empty($name)) {
        $error = 'Product name is required';
    } elseif ($price <= 0) {
        $error = 'Valid price is required';
    } else {
        // Handle image upload
        $imagePath = null;
        if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
            $uploadDir = UPLOAD_PATH . '/' . $storeId;
            $imagePath = handleUpload($_FILES['image'], $uploadDir);
        }

        if ($action === 'add') {
            $stmt = $pdo->prepare("
                INSERT INTO products (store_id, name, description, price, offer_price, image, stock_status, is_active, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, 1, NOW())
            ");
            $stmt->execute([$storeId, $name, $description, $price * 100, $offerPrice ? $offerPrice * 100 : null, $imagePath, $stockStatus]);
            flash('success', 'Product added successfully!');
            header('Location: products.php');
            exit;
        }
    }
}

// Get products
$stmt = $pdo->prepare("SELECT * FROM products WHERE store_id = ? ORDER BY created_at DESC");
$stmt->execute([$storeId]);
$products = $stmt->fetchAll();

include 'includes/header.php';
?>

<div class="dashboard-content">
    <?php if ($action === 'add'): ?>
        <!-- Add Product Form -->
        <div class="page-header">
            <h1>Add Product</h1>
            <a href="products.php" class="btn btn-outline">← Back</a>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-error"><?= e($error) ?></div>
        <?php endif; ?>

        <form method="POST" enctype="multipart/form-data" class="product-form">
            <div class="form-group">
                <label for="name">Product Name *</label>
                <input type="text" id="name" name="name" required
                    value="<?= e($_POST['name'] ?? '') ?>"
                    placeholder="Red Silk Saree">
            </div>

            <div class="form-row">
                <div class="form-group">
                    <label for="price">Price (₹) *</label>
                    <input type="number" id="price" name="price" required min="1"
                        value="<?= e($_POST['price'] ?? '') ?>"
                        placeholder="999">
                </div>

                <div class="form-group">
                    <label for="offer_price">Offer Price (₹)</label>
                    <input type="number" id="offer_price" name="offer_price" min="1"
                        value="<?= e($_POST['offer_price'] ?? '') ?>"
                        placeholder="799">
                    <small>Leave empty if no discount</small>
                </div>
            </div>

            <div class="form-group">
                <label for="description">Description</label>
                <textarea id="description" name="description" rows="3"
                    placeholder="Describe your product..."><?= e($_POST['description'] ?? '') ?></textarea>
            </div>

            <div class="form-group">
                <label for="image">Product Image</label>
                <input type="file" id="image" name="image" accept="image/*">
                <small>Max 5MB. JPG, PNG, or WebP.</small>
            </div>

            <div class="form-group">
                <label for="stock_status">Stock Status</label>
                <select id="stock_status" name="stock_status">
                    <option value="in_stock">In Stock</option>
                    <option value="out_of_stock">Out of Stock</option>
                </select>
            </div>

            <div class="form-actions">
                <button type="submit" class="btn btn-primary">Add Product</button>
            </div>
        </form>
    <?php else: ?>
        <!-- Products List -->
        <div class="page-header">
            <h1>Products</h1>
            <a href="products.php?action=add" class="btn btn-primary">+ Add Product</a>
        </div>

        <?php if ($success): ?>
            <div class="alert alert-success"><?= e($success) ?></div>
        <?php endif; ?>

        <?php if (empty($products)): ?>
            <div class="empty-state-small">
                <p>No products yet. Add your first product to start selling!</p>
                <a href="products.php?action=add" class="btn btn-primary">Add Product</a>
            </div>
        <?php else: ?>
            <div class="products-grid">
                <?php foreach ($products as $product): ?>
                    <div class="product-card">
                        <div class="product-image">
                            <?php if ($product['image']): ?>
                                <img src="/uploads/<?= $storeId ?>/<?= e($product['image']) ?>" alt="">
                            <?php else: ?>
                                <div class="no-image">📦</div>
                            <?php endif; ?>
                        </div>
                        <div class="product-info">
                            <h3><?= e($product['name']) ?></h3>
                            <div class="product-price">
                                <?php if ($product['offer_price']): ?>
                                    <span class="price-strike"><?= formatPrice($product['price']) ?></span>
                                    <span class="price-current"><?= formatPrice($product['offer_price']) ?></span>
                                <?php else: ?>
                                    <span class="price-current"><?= formatPrice($product['price']) ?></span>
                                <?php endif; ?>
                            </div>
                            <span class="stock-badge <?= $product['stock_status'] ?>">
                                <?= $product['stock_status'] === 'in_stock' ? 'In Stock' : 'Out of Stock' ?>
                            </span>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    <?php endif; ?>
</div>

<?php include 'includes/footer.php'; ?>