<?php

/**
 * Customer Storefront
 * {store}.my.ziteo.in
 */

require_once dirname(__DIR__, 2) . '/includes/db.php';
require_once dirname(__DIR__, 2) . '/includes/functions.php';

// Get store slug from subdomain or query param
$host = $_SERVER['HTTP_HOST'] ?? '';
$slug = '';

// Check if subdomain format: {slug}.my.ziteo.in
if (preg_match('/^([a-z0-9-]+)\.my\.ziteo\.in$/', $host, $matches)) {
    $slug = $matches[1];
} else {
    // Fallback to query param for local development
    $slug = $_GET['store'] ?? '';
}

if (empty($slug)) {
    // Show default page or redirect
    header('Location: https://ziteo.in');
    exit;
}

$pdo = db();

// Get store
$stmt = $pdo->prepare("
    SELECT s.*, u.name as owner_name
    FROM stores s
    JOIN users u ON s.user_id = u.id
    WHERE s.slug = ?
");
$stmt->execute([$slug]);
$store = $stmt->fetch();

if (!$store) {
    http_response_code(404);
    echo "<h1>Store not found</h1>";
    exit;
}

if ($store['status'] === 'pending') {
    echo "<h1>Coming Soon</h1><p>This store is being set up. Check back later!</p>";
    exit;
}

if ($store['status'] === 'suspended') {
    http_response_code(403);
    echo "<h1>Store Unavailable</h1>";
    exit;
}

// Get products
$stmt = $pdo->prepare("
    SELECT * FROM products 
    WHERE store_id = ? AND is_active = 1 
    ORDER BY created_at DESC
");
$stmt->execute([$store['id']]);
$products = $stmt->fetchAll();

// Theme config
$theme = json_decode($store['theme_config'] ?? '{}', true) ?: [];
$primaryColor = $theme['primary_color'] ?? '#25D366';
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= e($store['name']) ?></title>
    <meta name="description" content="Shop at <?= e($store['name']) ?>. Order on WhatsApp.">
    <link rel="stylesheet" href="assets/css/store.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary: <?= e($primaryColor) ?>;
        }
    </style>
</head>

<body>
    <!-- Header -->
    <header class="store-header">
        <div class="container">
            <h1 class="store-name"><?= e($store['name']) ?></h1>
            <a href="https://wa.me/91<?= e($store['whatsapp']) ?>" class="whatsapp-btn">
                💬 Chat with us
            </a>
        </div>
    </header>

    <!-- Products -->
    <main class="store-main">
        <div class="container">
            <?php if (empty($products)): ?>
                <div class="empty-store">
                    <p>No products available yet. Check back soon!</p>
                </div>
            <?php else: ?>
                <div class="products-grid">
                    <?php foreach ($products as $product): ?>
                        <div class="product-card">
                            <div class="product-image">
                                <?php if ($product['image']): ?>
                                    <img src="/uploads/<?= $store['id'] ?>/<?= e($product['image']) ?>"
                                        alt="<?= e($product['name']) ?>">
                                <?php else: ?>
                                    <div class="no-image">📦</div>
                                <?php endif; ?>

                                <?php if ($product['stock_status'] === 'out_of_stock'): ?>
                                    <span class="sold-badge">Sold Out</span>
                                <?php endif; ?>
                            </div>
                            <div class="product-info">
                                <h3><?= e($product['name']) ?></h3>
                                <div class="product-price">
                                    <?php if ($product['offer_price']): ?>
                                        <span class="price-strike"><?= formatPrice($product['price']) ?></span>
                                        <span class="price-current"><?= formatPrice($product['offer_price']) ?></span>
                                    <?php else: ?>
                                        <span class="price-current"><?= formatPrice($product['price']) ?></span>
                                    <?php endif; ?>
                                </div>

                                <?php if ($product['stock_status'] === 'in_stock'): ?>
                                    <?php
                                    $price = $product['offer_price'] ?: $product['price'];
                                    $message = urlencode("Hi, I want to order: {$product['name']} (" . formatPrice($price) . ")");
                                    ?>
                                    <a href="https://wa.me/91<?= e($store['whatsapp']) ?>?text=<?= $message ?>"
                                        class="buy-btn"
                                        onclick="logOrder(<?= $store['id'] ?>, <?= $product['id'] ?>, '<?= e($product['name']) ?>', <?= $price ?>)">
                                        Buy on WhatsApp
                                    </a>
                                <?php else: ?>
                                    <button class="buy-btn disabled" disabled>Sold Out</button>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </main>

    <!-- Footer -->
    <footer class="store-footer">
        <div class="container">
            <p>Powered by <a href="https://ziteo.in" target="_blank">ZITEO</a></p>
        </div>
    </footer>

    <script>
        function logOrder(storeId, productId, productName, price) {
            // Log order initiation
            fetch('/api/v1/orders.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    store_id: storeId,
                    product_id: productId,
                    product_name: productName,
                    price: price,
                    quantity: 1
                })
            }).catch(() => {}); // Silent fail - don't block WhatsApp redirect
        }
    </script>
</body>

</html>