<?php

/**
 * Product Settings Page
 * Configure which fields appear on product forms per category
 */
session_start();
require_once dirname(__DIR__, 2) . '/includes/auth.php';
requireAuth();

// Get store from URL parameter
$storeId = getSelectedStoreId();
if (!$storeId) {
    header('Location: dashboard.php');
    exit;
}

// Validate store ownership
$store = validateStoreOwnership($storeId);
if (!$store) {
    header('Location: dashboard.php');
    exit;
}

$pdo = db();
$error = '';
$success = flash('success');

// Available business categories (parent categories)
$businessCategories = [
    'general' => 'General',
    'clothing' => 'Clothing & Apparel',
    'electronics' => 'Electronics',
    'food' => 'Food & Beverages',
    'beauty' => 'Beauty & Personal Care',
    'home' => 'Home & Living',
    'jewelry' => 'Jewelry & Accessories'
];

// Current business category (from GET or default)
$currentCategory = $_GET['category'] ?? 'general';
if (!isset($businessCategories[$currentCategory])) {
    $currentCategory = 'general';
}

// Compulsory fields (always ON, cannot be toggled)
$compulsoryFields = [
    'product_name' => ['label' => 'Product Name', 'icon' => '📝'],
    'description' => ['label' => 'Description', 'icon' => '📄'],
    'images' => ['label' => 'Images', 'icon' => '🖼️'],
    'original_price' => ['label' => 'Original Price', 'icon' => '💰']
];

// Category-specific default toggleable fields
$categoryDefaultFields = [
    'general' => [
        'offer_price' => true,
        'product_category' => true,
        'color_options' => false,
        'size_options' => false
    ],
    'clothing' => [
        'offer_price' => true,
        'product_category' => true,
        'color_options' => true,
        'size_options' => true
    ],
    'electronics' => [
        'offer_price' => true,
        'product_category' => true,
        'color_options' => true,
        'size_options' => false
    ],
    'food' => [
        'offer_price' => true,
        'product_category' => true,
        'color_options' => false,
        'size_options' => false
    ],
    'beauty' => [
        'offer_price' => true,
        'product_category' => true,
        'color_options' => true,
        'size_options' => true
    ],
    'home' => [
        'offer_price' => true,
        'product_category' => true,
        'color_options' => true,
        'size_options' => true
    ],
    'jewelry' => [
        'offer_price' => true,
        'product_category' => true,
        'color_options' => true,
        'size_options' => true
    ]
];

// Toggleable fields configuration
$toggleableFields = [
    'offer_price' => ['label' => 'Offer Price', 'icon' => '🏷️', 'description' => 'Add discounted/sale price for products'],
    'product_category' => ['label' => 'Category', 'icon' => '📂', 'description' => 'Organize products by category'],
    'color_options' => ['label' => 'Color Options', 'icon' => '🎨', 'description' => 'Add color selection for products'],
    'size_options' => ['label' => 'Size Options', 'icon' => '📏', 'description' => 'Add size selection for products']
];

// Maximum total fields (compulsory + toggleable + custom)
$maxTotalFields = 10;
$compulsoryCount = count($compulsoryFields);
$maxCustomFields = $maxTotalFields - $compulsoryCount;

// Load existing settings
function getFieldSettings($pdo, $storeId, $category)
{
    $stmt = $pdo->prepare("SELECT field_config FROM product_field_settings WHERE store_id = ? AND category = ?");
    $stmt->execute([$storeId, $category]);
    $result = $stmt->fetchColumn();
    return $result ? json_decode($result, true) : null;
}

// Save settings
function saveFieldSettings($pdo, $storeId, $category, $config)
{
    $stmt = $pdo->prepare("
        INSERT INTO product_field_settings (store_id, category, field_config, created_at)
        VALUES (?, ?, ?, NOW())
        ON DUPLICATE KEY UPDATE field_config = VALUES(field_config), updated_at = NOW()
    ");
    return $stmt->execute([$storeId, $category, json_encode($config)]);
}

// Get predefined categories for current business category
function getPredefinedCategories($pdo, $parentCategory)
{
    $stmt = $pdo->prepare("SELECT * FROM product_categories WHERE parent_category = ? AND store_id IS NULL AND is_active = 1 ORDER BY name");
    $stmt->execute([$parentCategory]);
    return $stmt->fetchAll();
}

// Get store's custom categories
function getStoreCategories($pdo, $storeId, $parentCategory)
{
    $stmt = $pdo->prepare("SELECT * FROM product_categories WHERE store_id = ? AND parent_category = ? AND is_active = 1 ORDER BY name");
    $stmt->execute([$storeId, $parentCategory]);
    return $stmt->fetchAll();
}

// Get store's selected categories (from settings)
function getSelectedCategories($settings)
{
    return $settings['selected_categories'] ?? [];
}

// Load current settings or use defaults
$settings = getFieldSettings($pdo, $storeId, $currentCategory);
if (!$settings) {
    $defaultFields = $categoryDefaultFields[$currentCategory] ?? $categoryDefaultFields['general'];
    $settings = [
        'fields' => [],
        'custom_fields' => [],
        'selected_categories' => []
    ];
    foreach ($defaultFields as $key => $enabled) {
        $settings['fields'][$key] = ['enabled' => $enabled];
    }
}

// Ensure selected_categories exists
if (!isset($settings['selected_categories'])) {
    $settings['selected_categories'] = [];
}

// Load categories from database
$predefinedCategories = [];
$storeCategories = [];
try {
    $predefinedCategories = getPredefinedCategories($pdo, $currentCategory);
    $storeCategories = getStoreCategories($pdo, $storeId, $currentCategory);
} catch (PDOException $e) {
    // Table might not exist yet - silently fail
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? 'save';

    if ($action === 'save') {
        $newSettings = [
            'fields' => [],
            'custom_fields' => $settings['custom_fields'] ?? [],
            'selected_categories' => $settings['selected_categories'] ?? []
        ];

        foreach ($toggleableFields as $key => $field) {
            $newSettings['fields'][$key] = [
                'enabled' => isset($_POST['field_' . $key])
            ];
        }

        if (saveFieldSettings($pdo, $storeId, $currentCategory, $newSettings)) {
            flash('success', 'Product settings saved successfully!');
            header("Location: product-settings.php?store=$storeId&category=$currentCategory");
            exit;
        } else {
            $error = 'Failed to save settings. Please try again.';
        }
    } elseif ($action === 'add_custom_field') {
        $fieldName = trim($_POST['custom_field_name'] ?? '');
        $fieldType = $_POST['custom_field_type'] ?? 'text';

        $customFields = $settings['custom_fields'] ?? [];
        $currentTotal = $compulsoryCount + count($customFields);

        if (empty($fieldName)) {
            $error = 'Field name is required.';
        } elseif (strlen($fieldName) > 50) {
            $error = 'Field name must be 50 characters or less.';
        } elseif ($currentTotal >= $maxTotalFields) {
            $error = "Maximum of $maxTotalFields fields reached.";
        } else {
            $duplicate = false;
            foreach ($customFields as $cf) {
                if (strtolower($cf['name']) === strtolower($fieldName)) {
                    $duplicate = true;
                    break;
                }
            }

            if ($duplicate) {
                $error = 'A custom field with this name already exists.';
            } else {
                $customFields[] = [
                    'name' => $fieldName,
                    'type' => $fieldType,
                    'required' => false
                ];

                $settings['custom_fields'] = $customFields;
                if (saveFieldSettings($pdo, $storeId, $currentCategory, $settings)) {
                    flash('success', "Custom field '$fieldName' added successfully!");
                    header("Location: product-settings.php?store=$storeId&category=$currentCategory");
                    exit;
                } else {
                    $error = 'Failed to add custom field.';
                }
            }
        }
    } elseif ($action === 'delete_custom_field') {
        $fieldIndex = (int)($_POST['field_index'] ?? -1);
        $customFields = $settings['custom_fields'] ?? [];

        if ($fieldIndex >= 0 && $fieldIndex < count($customFields)) {
            $deletedName = $customFields[$fieldIndex]['name'];
            array_splice($customFields, $fieldIndex, 1);
            $settings['custom_fields'] = $customFields;

            if (saveFieldSettings($pdo, $storeId, $currentCategory, $settings)) {
                flash('success', "Custom field '$deletedName' deleted.");
                header("Location: product-settings.php?store=$storeId&category=$currentCategory");
                exit;
            } else {
                $error = 'Failed to delete custom field.';
            }
        }
    } elseif ($action === 'add_category') {
        $categoryId = (int)($_POST['category_id'] ?? 0);
        $selectedCategories = $settings['selected_categories'] ?? [];

        if ($categoryId > 0 && !in_array($categoryId, $selectedCategories)) {
            $selectedCategories[] = $categoryId;
            $settings['selected_categories'] = $selectedCategories;

            if (saveFieldSettings($pdo, $storeId, $currentCategory, $settings)) {
                flash('success', 'Category added successfully!');
                header("Location: product-settings.php?store=$storeId&category=$currentCategory");
                exit;
            }
        }
    } elseif ($action === 'remove_category') {
        $categoryId = (int)($_POST['category_id'] ?? 0);
        $selectedCategories = $settings['selected_categories'] ?? [];

        $selectedCategories = array_filter($selectedCategories, fn($id) => $id !== $categoryId);
        $settings['selected_categories'] = array_values($selectedCategories);

        if (saveFieldSettings($pdo, $storeId, $currentCategory, $settings)) {
            flash('success', 'Category removed.');
            header("Location: product-settings.php?store=$storeId&category=$currentCategory");
            exit;
        }
    } elseif ($action === 'add_custom_category') {
        $categoryName = trim($_POST['custom_category_name'] ?? '');

        if (empty($categoryName)) {
            $error = 'Category name is required.';
        } elseif (strlen($categoryName) > 100) {
            $error = 'Category name must be 100 characters or less.';
        } else {
            $slug = strtolower(preg_replace('/[^a-zA-Z0-9]+/', '-', $categoryName));

            try {
                $stmt = $pdo->prepare("
                    INSERT INTO product_categories (store_id, parent_category, name, slug, is_custom, created_at)
                    VALUES (?, ?, ?, ?, 1, NOW())
                ");
                $stmt->execute([$storeId, $currentCategory, $categoryName, $slug]);
                $newCategoryId = $pdo->lastInsertId();

                // Auto-select the new category
                $selectedCategories = $settings['selected_categories'] ?? [];
                $selectedCategories[] = (int)$newCategoryId;
                $settings['selected_categories'] = $selectedCategories;
                saveFieldSettings($pdo, $storeId, $currentCategory, $settings);

                flash('success', "Custom category '$categoryName' created and added!");
                header("Location: product-settings.php?store=$storeId&category=$currentCategory");
                exit;
            } catch (PDOException $e) {
                $error = 'Failed to create custom category.';
            }
        }
    } elseif ($action === 'delete_custom_category') {
        $categoryId = (int)($_POST['category_id'] ?? 0);

        try {
            // Only allow deleting custom categories for this store
            $stmt = $pdo->prepare("DELETE FROM product_categories WHERE id = ? AND store_id = ? AND is_custom = 1");
            $stmt->execute([$categoryId, $storeId]);

            // Remove from selected categories
            $selectedCategories = array_filter($settings['selected_categories'] ?? [], fn($id) => $id !== $categoryId);
            $settings['selected_categories'] = array_values($selectedCategories);
            saveFieldSettings($pdo, $storeId, $currentCategory, $settings);

            flash('success', 'Custom category deleted.');
            header("Location: product-settings.php?store=$storeId&category=$currentCategory");
            exit;
        } catch (PDOException $e) {
            $error = 'Failed to delete category.';
        }
    }

    // Reload settings after changes
    $settings = getFieldSettings($pdo, $storeId, $currentCategory) ?? $settings;
}

// Calculate field counts (compulsory + enabled optional + custom)
$customFieldCount = count($settings['custom_fields'] ?? []);
$enabledOptionalCount = 0;
foreach ($toggleableFields as $key => $field) {
    if ($settings['fields'][$key]['enabled'] ?? false) {
        $enabledOptionalCount++;
    }
}
$totalFieldCount = $compulsoryCount + $enabledOptionalCount + $customFieldCount;
$remainingFields = $maxTotalFields - $totalFieldCount;

// Check if category field is enabled
$isCategoryEnabled = $settings['fields']['product_category']['enabled'] ?? false;

// Get all categories (predefined + store custom) with selected status
$allCategories = [];
$selectedCategoryIds = $settings['selected_categories'] ?? [];

foreach ($predefinedCategories as $cat) {
    $cat['is_selected'] = in_array($cat['id'], $selectedCategoryIds);
    $allCategories[] = $cat;
}
foreach ($storeCategories as $cat) {
    $cat['is_selected'] = in_array($cat['id'], $selectedCategoryIds);
    $allCategories[] = $cat;
}

include 'includes/header.php';
?>

<div class="dashboard-content">
    <div class="page-header">
        <div>
            <h1>Product Settings</h1>
            <p class="subtitle">Configure which fields appear when adding products</p>
        </div>
        <a href="dashboard.php?store=<?= $storeId ?>" class="btn btn-outline">← Back to Dashboard</a>
    </div>

    <?php if ($error): ?>
        <div class="alert alert-error"><?= e($error) ?></div>
    <?php endif; ?>

    <?php if ($success): ?>
        <div class="alert alert-success"><?= e($success) ?></div>
    <?php endif; ?>

    <!-- Business Category Selector -->
    <div class="settings-section">
        <h3>📂 Business Category</h3>
        <p class="section-description">Select your store's primary business category to see recommended field settings.</p>
        <div class="category-selector">
            <?php foreach ($businessCategories as $key => $label): ?>
                <a href="product-settings.php?store=<?= $storeId ?>&category=<?= $key ?>"
                    class="category-chip <?= $key === $currentCategory ? 'active' : '' ?>">
                    <?= e($label) ?>
                </a>
            <?php endforeach; ?>
        </div>
    </div>

    <!-- Field Count Indicator -->
    <div class="field-count-bar">
        <div class="field-count-info">
            <span class="field-count-number"><?= $totalFieldCount ?>/<?= $maxTotalFields ?></span>
            <span class="field-count-label">fields used</span>
        </div>
        <div class="field-count-progress">
            <div class="field-count-fill" style="width: <?= ($totalFieldCount / $maxTotalFields) * 100 ?>%"></div>
        </div>
    </div>

    <form method="POST" id="settingsForm">
        <input type="hidden" name="action" value="save">

        <!-- Save Button (Top) -->
        <div class="settings-actions-top">
            <button type="submit" class="btn btn-primary btn-large">💾 Save Settings</button>
            <span class="field-summary">📊 <?= $compulsoryCount ?> compulsory + <?= $enabledOptionalCount ?> optional + <?= $customFieldCount ?> custom = <?= $totalFieldCount ?> fields</span>
        </div>

        <!-- Compulsory Fields -->
        <div class="settings-section">
            <h3>🔒 Compulsory Fields</h3>
            <p class="section-description">These fields are required for all products and cannot be turned off.</p>
            <div class="fields-grid">
                <?php foreach ($compulsoryFields as $key => $field): ?>
                    <div class="field-card compulsory">
                        <div class="field-header">
                            <span class="field-icon"><?= $field['icon'] ?></span>
                            <span class="field-name"><?= e($field['label']) ?></span>
                        </div>
                        <div class="field-toggle">
                            <span class="toggle-status always-on">Always ON</span>
                            <div class="toggle-switch locked">
                                <input type="checkbox" checked disabled>
                                <span class="toggle-slider"></span>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>

        <!-- Toggleable Fields -->
        <div class="settings-section">
            <h3>⚙️ Optional Fields</h3>
            <p class="section-description">Toggle these fields On or Off based on your product requirements.</p>
            <div class="fields-grid">
                <?php foreach ($toggleableFields as $key => $field): ?>
                    <?php $isEnabled = $settings['fields'][$key]['enabled'] ?? false; ?>
                    <div class="field-card" id="field-card-<?= $key ?>">
                        <div class="field-header">
                            <span class="field-icon"><?= $field['icon'] ?></span>
                            <div>
                                <span class="field-name"><?= e($field['label']) ?></span>
                                <span class="field-description"><?= e($field['description']) ?></span>
                            </div>
                        </div>
                        <div class="field-toggle">
                            <span class="toggle-status <?= $isEnabled ? 'on' : 'off' ?>"><?= $isEnabled ? 'ON' : 'OFF' ?></span>
                            <label class="toggle-switch">
                                <input type="checkbox" name="field_<?= $key ?>" <?= $isEnabled ? 'checked' : '' ?>
                                    onchange="updateToggleStatus(this, '<?= $key ?>')">
                                <span class="toggle-slider"></span>
                            </label>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>

        <!-- Save Button (Bottom) -->
        <div class="settings-actions">
            <button type="submit" class="btn btn-primary">💾 Save Settings</button>
        </div>
    </form>

    <!-- Category Management Section (shown when Category is enabled) -->
    <div class="settings-section category-management" id="categoryManagement" style="<?= $isCategoryEnabled ? '' : 'display: none;' ?>">
        <h3>📁 Manage Product Categories</h3>
        <p class="section-description">Add categories to organize your products. Search from predefined options or create custom categories.</p>

        <!-- Selected Categories -->
        <div class="selected-categories">
            <h4>Selected Categories</h4>
            <?php
            $selectedCats = array_filter($allCategories, fn($c) => $c['is_selected']);
            if (!empty($selectedCats)):
            ?>
                <div class="category-tags">
                    <?php foreach ($selectedCats as $cat): ?>
                        <div class="category-tag">
                            <span><?= e($cat['name']) ?></span>
                            <form method="POST" class="inline-form">
                                <input type="hidden" name="action" value="remove_category">
                                <input type="hidden" name="category_id" value="<?= $cat['id'] ?>">
                                <button type="submit" class="tag-remove" title="Remove">×</button>
                            </form>
                            <?php if ($cat['is_custom']): ?>
                                <form method="POST" class="inline-form" onsubmit="return confirm('Delete this custom category permanently?');">
                                    <input type="hidden" name="action" value="delete_custom_category">
                                    <input type="hidden" name="category_id" value="<?= $cat['id'] ?>">
                                    <button type="submit" class="tag-delete" title="Delete permanently">🗑️</button>
                                </form>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php else: ?>
                <p class="no-categories">No categories selected yet. Add categories below.</p>
            <?php endif; ?>
        </div>

        <!-- Add Category -->
        <div class="add-category-section">
            <h4>➕ Add Category</h4>

            <!-- Search/Select from Predefined -->
            <div class="category-search-box">
                <label>Select from predefined categories:</label>
                <div class="category-options">
                    <?php
                    $availableCats = array_filter($allCategories, fn($c) => !$c['is_selected'] && !$c['is_custom']);
                    if (!empty($availableCats)):
                    ?>
                        <?php foreach ($availableCats as $cat): ?>
                            <form method="POST" class="inline-form">
                                <input type="hidden" name="action" value="add_category">
                                <input type="hidden" name="category_id" value="<?= $cat['id'] ?>">
                                <button type="submit" class="category-option-btn">
                                    + <?= e($cat['name']) ?>
                                </button>
                            </form>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <p class="all-added">All predefined categories have been added.</p>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Create Custom Category -->
            <div class="create-custom-category">
                <label>Or create a custom category:</label>
                <form method="POST" class="custom-category-form">
                    <input type="hidden" name="action" value="add_custom_category">
                    <input type="text" name="custom_category_name" placeholder="Enter custom category name" maxlength="100" required>
                    <button type="submit" class="btn btn-primary">Create & Add</button>
                </form>
            </div>
        </div>
    </div>

    <!-- Custom Fields Section -->
    <div class="settings-section">
        <h3>✨ Custom Fields</h3>
        <p class="section-description">Create custom fields for additional product information. You can add up to <?= $maxCustomFields ?> custom fields.</p>

        <?php if (!empty($settings['custom_fields'])): ?>
            <div class="custom-fields-list">
                <?php foreach ($settings['custom_fields'] as $index => $customField): ?>
                    <div class="custom-field-item">
                        <div class="custom-field-info">
                            <span class="custom-field-icon">📋</span>
                            <div>
                                <span class="custom-field-name"><?= e($customField['name']) ?></span>
                                <span class="custom-field-type"><?= ucfirst($customField['type']) ?></span>
                            </div>
                        </div>
                        <form method="POST" class="delete-form" onsubmit="return confirm('Delete this custom field?');">
                            <input type="hidden" name="action" value="delete_custom_field">
                            <input type="hidden" name="field_index" value="<?= $index ?>">
                            <button type="submit" class="btn-delete" title="Delete field">🗑️</button>
                        </form>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <p class="no-custom-fields">No custom fields added yet.</p>
        <?php endif; ?>

        <?php if ($remainingFields > 0): ?>
            <div class="add-custom-field-form">
                <h4>➕ Add New Custom Field</h4>
                <form method="POST" class="custom-field-form">
                    <input type="hidden" name="action" value="add_custom_field">
                    <div class="form-row">
                        <div class="form-group">
                            <label for="custom_field_name">Field Name</label>
                            <input type="text" id="custom_field_name" name="custom_field_name"
                                placeholder="e.g., Material, Weight, Brand" maxlength="50" required>
                        </div>
                        <div class="form-group">
                            <label for="custom_field_type">Field Type</label>
                            <select id="custom_field_type" name="custom_field_type">
                                <option value="text">Text</option>
                                <option value="number">Number</option>
                                <option value="textarea">Long Text</option>
                                <option value="select">Dropdown</option>
                            </select>
                        </div>
                    </div>
                    <button type="submit" class="btn btn-primary">Add Field</button>
                    <span class="remaining-fields"><?= $remainingFields ?> field(s) remaining</span>
                </form>
            </div>
        <?php else: ?>
            <div class="limit-reached">
                <p>⚠️ Maximum field limit reached. Delete a custom field to add a new one.</p>
            </div>
        <?php endif; ?>
    </div>
</div>

<script>
    function updateToggleStatus(checkbox, fieldKey) {
        const statusSpan = checkbox.closest('.field-toggle').querySelector('.toggle-status');
        if (checkbox.checked) {
            statusSpan.textContent = 'ON';
            statusSpan.className = 'toggle-status on';
        } else {
            statusSpan.textContent = 'OFF';
            statusSpan.className = 'toggle-status off';
        }

        // Show/hide category management section based on product_category toggle
        if (fieldKey === 'product_category') {
            const categorySection = document.getElementById('categoryManagement');
            if (categorySection) {
                categorySection.style.display = checkbox.checked ? 'block' : 'none';
            }
        }
    }
</script>

<?php include 'includes/footer.php'; ?>