<?php

/**
 * Products Management Page
 * With dynamic fields based on Product Settings
 */
session_start();
require_once dirname(__DIR__, 2) . '/includes/auth.php';
requireAuth();

// Get store from URL parameter
$storeId = getSelectedStoreId();
if (!$storeId) {
    header('Location: dashboard.php');
    exit;
}

// Validate store ownership
$store = validateStoreOwnership($storeId);
if (!$store) {
    header('Location: dashboard.php');
    exit;
}

$pdo = db();
$action = $_GET['action'] ?? 'list';
$error = '';
$success = flash('success');

// Get store's business category from theme_config or default to 'general'
$storeConfig = json_decode($store['theme_config'] ?? '{}', true);
$businessCategory = $storeConfig['business_category'] ?? 'general';

// Load product field settings for this store
function getProductFieldSettings($pdo, $storeId, $category)
{
    try {
        $stmt = $pdo->prepare("SELECT field_config FROM product_field_settings WHERE store_id = ? AND category = ?");
        $stmt->execute([$storeId, $category]);
        $result = $stmt->fetchColumn();
        return $result ? json_decode($result, true) : null;
    } catch (PDOException $e) {
        return null;
    }
}

// Get available categories for dropdown
function getAvailableCategories($pdo, $storeId, $businessCategory, $settings)
{
    $selectedCategoryIds = $settings['selected_categories'] ?? [];
    if (empty($selectedCategoryIds)) {
        return [];
    }

    try {
        $placeholders = implode(',', array_fill(0, count($selectedCategoryIds), '?'));
        $stmt = $pdo->prepare("SELECT * FROM product_categories WHERE id IN ($placeholders) AND is_active = 1 ORDER BY name");
        $stmt->execute($selectedCategoryIds);
        return $stmt->fetchAll();
    } catch (PDOException $e) {
        return [];
    }
}

// Load settings
$fieldSettings = getProductFieldSettings($pdo, $storeId, $businessCategory);
if (!$fieldSettings) {
    // Default settings if none configured
    $fieldSettings = [
        'fields' => [
            'offer_price' => ['enabled' => true],
            'product_category' => ['enabled' => false],
            'color_options' => ['enabled' => false],
            'size_options' => ['enabled' => false]
        ],
        'custom_fields' => [],
        'selected_categories' => []
    ];
}

// Get available categories if category field is enabled
$availableCategories = [];
if ($fieldSettings['fields']['product_category']['enabled'] ?? false) {
    $availableCategories = getAvailableCategories($pdo, $storeId, $businessCategory, $fieldSettings);
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $price = (int)($_POST['price'] ?? 0);
    $stockStatus = $_POST['stock_status'] ?? 'in_stock';

    // Optional fields based on settings
    $offerPrice = null;
    if ($fieldSettings['fields']['offer_price']['enabled'] ?? false) {
        $offerPrice = !empty($_POST['offer_price']) ? (int)$_POST['offer_price'] : null;
    }

    $categoryId = null;
    if ($fieldSettings['fields']['product_category']['enabled'] ?? false) {
        $categoryId = !empty($_POST['category_id']) ? (int)$_POST['category_id'] : null;
    }

    // Build variants JSON (colors and sizes with price adjustments)
    $variants = ['colors' => [], 'sizes' => []];

    if ($fieldSettings['fields']['color_options']['enabled'] ?? false) {
        $colorNames = $_POST['color_name'] ?? [];
        $colorPrices = $_POST['color_price'] ?? [];
        foreach ($colorNames as $i => $colorName) {
            $colorName = trim($colorName);
            if (!empty($colorName)) {
                $variants['colors'][] = [
                    'name' => $colorName,
                    'price_adjustment' => (int)($colorPrices[$i] ?? 0) * 100 // Convert to paise
                ];
            }
        }
    }

    if ($fieldSettings['fields']['size_options']['enabled'] ?? false) {
        $sizeNames = $_POST['size_name'] ?? [];
        $sizePrices = $_POST['size_price'] ?? [];
        foreach ($sizeNames as $i => $sizeName) {
            $sizeName = trim($sizeName);
            if (!empty($sizeName)) {
                $variants['sizes'][] = [
                    'name' => $sizeName,
                    'price_adjustment' => (int)($sizePrices[$i] ?? 0) * 100 // Convert to paise
                ];
            }
        }
    }

    // Build custom data JSON
    $customData = [];
    foreach ($fieldSettings['custom_fields'] ?? [] as $customField) {
        $fieldKey = 'custom_' . strtolower(preg_replace('/[^a-zA-Z0-9]/', '_', $customField['name']));
        $customData[$customField['name']] = trim($_POST[$fieldKey] ?? '');
    }

    // Validation
    if (empty($name)) {
        $error = 'Product name is required';
    } elseif ($price <= 0) {
        $error = 'Valid price is required';
    } else {
        // Handle image upload
        $imagePath = null;
        if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
            $uploadDir = UPLOAD_PATH . '/' . $storeId;
            $imagePath = handleUpload($_FILES['image'], $uploadDir);
        }

        if ($action === 'add') {
            // Check product limit
            $stmt = $pdo->prepare("SELECT p.product_limit FROM plans p JOIN users u ON u.plan_id = p.id WHERE u.id = ?");
            $stmt->execute([currentUserId()]);
            $productLimit = $stmt->fetchColumn() ?: 10;

            $stmt = $pdo->prepare("SELECT COUNT(*) FROM products WHERE store_id = ?");
            $stmt->execute([$storeId]);
            $currentProducts = $stmt->fetchColumn();

            if ($currentProducts >= $productLimit) {
                $error = "You have reached the maximum number of products ($productLimit) for your plan. <a href='select_plan.php' class='underline'>Upgrade Plan</a> to add more.";
            } else {
                // Convert variants and custom_data to JSON
                $variantsJson = json_encode($variants);
                $customDataJson = !empty($customData) ? json_encode($customData) : null;

                $stmt = $pdo->prepare("
                    INSERT INTO products (store_id, category_id, name, description, price, offer_price, variants, custom_data, image, stock_status, is_active, created_at)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1, NOW())
                ");
                $stmt->execute([
                    $storeId,
                    $categoryId,
                    $name,
                    $description,
                    $price * 100,
                    $offerPrice ? $offerPrice * 100 : null,
                    $variantsJson,
                    $customDataJson,
                    $imagePath,
                    $stockStatus
                ]);
                flash('success', 'Product added successfully!');
                header("Location: products.php?store=$storeId");
                exit;
            }
        }
    }
}

// Get products with category names
$stmt = $pdo->prepare("
    SELECT p.*, pc.name as category_name 
    FROM products p 
    LEFT JOIN product_categories pc ON p.category_id = pc.id 
    WHERE p.store_id = ? 
    ORDER BY p.created_at DESC
");
$stmt->execute([$storeId]);
$products = $stmt->fetchAll();

include 'includes/header.php';
?>

<div class="dashboard-content">
    <?php if ($action === 'add'): ?>
        <!-- Add Product Form -->
        <div class="page-header">
            <h1>Add Product</h1>
            <a href="products.php?store=<?= $storeId ?>" class="btn btn-outline">← Back</a>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-error"><?= $error ?></div>
        <?php endif; ?>

        <form method="POST" enctype="multipart/form-data" class="product-form dynamic-product-form">
            <!-- Compulsory: Product Name -->
            <div class="form-group">
                <label for="name">Product Name *</label>
                <input type="text" id="name" name="name" required
                    value="<?= e($_POST['name'] ?? '') ?>"
                    placeholder="Enter product name">
            </div>

            <!-- Compulsory: Description -->
            <div class="form-group">
                <label for="description">Description *</label>
                <textarea id="description" name="description" rows="3"
                    placeholder="Describe your product..."><?= e($_POST['description'] ?? '') ?></textarea>
            </div>

            <!-- Compulsory: Images -->
            <div class="form-group">
                <label for="image">Product Image *</label>
                <input type="file" id="image" name="image" accept="image/*">
                <small>Max 5MB. JPG, PNG, or WebP.</small>
            </div>

            <!-- Compulsory: Original Price -->
            <div class="form-row">
                <div class="form-group">
                    <label for="price">Original Price (₹) *</label>
                    <input type="number" id="price" name="price" required min="1"
                        value="<?= e($_POST['price'] ?? '') ?>"
                        placeholder="999">
                </div>

                <?php if ($fieldSettings['fields']['offer_price']['enabled'] ?? false): ?>
                    <!-- Dynamic: Offer Price -->
                    <div class="form-group">
                        <label for="offer_price">Offer Price (₹)</label>
                        <input type="number" id="offer_price" name="offer_price" min="1"
                            value="<?= e($_POST['offer_price'] ?? '') ?>"
                            placeholder="799">
                        <small>Leave empty if no discount</small>
                    </div>
                <?php endif; ?>
            </div>

            <?php if ($fieldSettings['fields']['product_category']['enabled'] ?? false): ?>
                <!-- Dynamic: Category -->
                <div class="form-group">
                    <label for="category_id">Category</label>
                    <?php if (!empty($availableCategories)): ?>
                        <select id="category_id" name="category_id">
                            <option value="">-- Select Category --</option>
                            <?php foreach ($availableCategories as $cat): ?>
                                <option value="<?= $cat['id'] ?>" <?= ($_POST['category_id'] ?? '') == $cat['id'] ? 'selected' : '' ?>>
                                    <?= e($cat['name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    <?php else: ?>
                        <p class="field-notice">No categories configured. <a href="product-settings.php?store=<?= $storeId ?>">Add categories</a> in Product Settings.</p>
                    <?php endif; ?>
                </div>
            <?php endif; ?>

            <?php if ($fieldSettings['fields']['color_options']['enabled'] ?? false): ?>
                <!-- Dynamic: Color Options with Price Variants -->
                <div class="form-group variant-group">
                    <label>Color Options (with price adjustment)</label>
                    <div class="variants-container" id="colorVariants">
                        <div class="variant-row">
                            <input type="text" name="color_name[]" placeholder="Color name (e.g., Red)">
                            <input type="number" name="color_price[]" placeholder="Price adjustment (₹)" min="0" value="0">
                            <button type="button" class="btn-remove-variant" onclick="removeVariant(this)">×</button>
                        </div>
                    </div>
                    <button type="button" class="btn btn-sm btn-outline" onclick="addColorVariant()">+ Add Color</button>
                    <small>Price adjustment adds to base price. Leave 0 for no adjustment.</small>
                </div>
            <?php endif; ?>

            <?php if ($fieldSettings['fields']['size_options']['enabled'] ?? false): ?>
                <!-- Dynamic: Size Options with Price Variants -->
                <div class="form-group variant-group">
                    <label>Size Options (with price adjustment)</label>
                    <div class="variants-container" id="sizeVariants">
                        <div class="variant-row">
                            <input type="text" name="size_name[]" placeholder="Size (e.g., S, M, L, XL)">
                            <input type="number" name="size_price[]" placeholder="Price adjustment (₹)" min="0" value="0">
                            <button type="button" class="btn-remove-variant" onclick="removeVariant(this)">×</button>
                        </div>
                    </div>
                    <button type="button" class="btn btn-sm btn-outline" onclick="addSizeVariant()">+ Add Size</button>
                    <small>Price adjustment adds to base price. Leave 0 for no adjustment.</small>
                </div>
            <?php endif; ?>

            <?php if (!empty($fieldSettings['custom_fields'])): ?>
                <!-- Custom Fields -->
                <div class="custom-fields-section">
                    <h3>Additional Information</h3>
                    <?php foreach ($fieldSettings['custom_fields'] as $customField): ?>
                        <?php
                        $fieldKey = 'custom_' . strtolower(preg_replace('/[^a-zA-Z0-9]/', '_', $customField['name']));
                        $fieldType = $customField['type'] ?? 'text';
                        ?>
                        <div class="form-group">
                            <label for="<?= $fieldKey ?>"><?= e($customField['name']) ?></label>
                            <?php if ($fieldType === 'textarea'): ?>
                                <textarea id="<?= $fieldKey ?>" name="<?= $fieldKey ?>" rows="3"
                                    placeholder="Enter <?= e($customField['name']) ?>"><?= e($_POST[$fieldKey] ?? '') ?></textarea>
                            <?php elseif ($fieldType === 'number'): ?>
                                <input type="number" id="<?= $fieldKey ?>" name="<?= $fieldKey ?>"
                                    value="<?= e($_POST[$fieldKey] ?? '') ?>"
                                    placeholder="Enter <?= e($customField['name']) ?>">
                            <?php elseif ($fieldType === 'select'): ?>
                                <input type="text" id="<?= $fieldKey ?>" name="<?= $fieldKey ?>"
                                    value="<?= e($_POST[$fieldKey] ?? '') ?>"
                                    placeholder="Enter <?= e($customField['name']) ?>">
                            <?php else: ?>
                                <input type="text" id="<?= $fieldKey ?>" name="<?= $fieldKey ?>"
                                    value="<?= e($_POST[$fieldKey] ?? '') ?>"
                                    placeholder="Enter <?= e($customField['name']) ?>">
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>

            <div class="form-group">
                <label for="stock_status">Stock Status</label>
                <select id="stock_status" name="stock_status">
                    <option value="in_stock">In Stock</option>
                    <option value="out_of_stock">Out of Stock</option>
                </select>
            </div>

            <div class="form-actions">
                <button type="submit" class="btn btn-primary">Add Product</button>
            </div>
        </form>
    <?php else: ?>
        <!-- Products List -->
        <div class="page-header">
            <h1>Products</h1>
            <a href="products.php?store=<?= $storeId ?>&action=add" class="btn btn-primary">+ Add Product</a>
        </div>

        <?php if ($success): ?>
            <div class="alert alert-success"><?= e($success) ?></div>
        <?php endif; ?>

        <?php if (empty($products)): ?>
            <div class="empty-state-small">
                <p>No products yet. Add your first product to start selling!</p>
                <a href="products.php?store=<?= $storeId ?>&action=add" class="btn btn-primary">Add Product</a>
            </div>
        <?php else: ?>
            <div class="products-grid">
                <?php foreach ($products as $product): ?>
                    <div class="product-card">
                        <div class="product-image">
                            <?php if ($product['image']): ?>
                                <img src="/uploads/<?= $storeId ?>/<?= e($product['image']) ?>" alt="">
                            <?php else: ?>
                                <div class="no-image">📦</div>
                            <?php endif; ?>
                        </div>
                        <div class="product-info">
                            <h3><?= e($product['name']) ?></h3>
                            <?php if (!empty($product['category_name'])): ?>
                                <span class="product-category-badge"><?= e($product['category_name']) ?></span>
                            <?php endif; ?>
                            <div class="product-price">
                                <?php if ($product['offer_price']): ?>
                                    <span class="price-strike"><?= formatPrice($product['price']) ?></span>
                                    <span class="price-current"><?= formatPrice($product['offer_price']) ?></span>
                                <?php else: ?>
                                    <span class="price-current"><?= formatPrice($product['price']) ?></span>
                                <?php endif; ?>
                            </div>
                            <?php
                            // Show variant badges if any
                            $variants = json_decode($product['variants'] ?? '{}', true);
                            if (!empty($variants['colors']) || !empty($variants['sizes'])):
                            ?>
                                <div class="variant-badges">
                                    <?php if (!empty($variants['colors'])): ?>
                                        <span class="variant-badge">🎨 <?= count($variants['colors']) ?> colors</span>
                                    <?php endif; ?>
                                    <?php if (!empty($variants['sizes'])): ?>
                                        <span class="variant-badge">📏 <?= count($variants['sizes']) ?> sizes</span>
                                    <?php endif; ?>
                                </div>
                            <?php endif; ?>
                            <span class="stock-badge <?= $product['stock_status'] ?>">
                                <?= $product['stock_status'] === 'in_stock' ? 'In Stock' : 'Out of Stock' ?>
                            </span>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    <?php endif; ?>
</div>

<script>
    function addColorVariant() {
        const container = document.getElementById('colorVariants');
        const row = document.createElement('div');
        row.className = 'variant-row';
        row.innerHTML = `
        <input type="text" name="color_name[]" placeholder="Color name (e.g., Blue)">
        <input type="number" name="color_price[]" placeholder="Price adjustment (₹)" min="0" value="0">
        <button type="button" class="btn-remove-variant" onclick="removeVariant(this)">×</button>
    `;
        container.appendChild(row);
    }

    function addSizeVariant() {
        const container = document.getElementById('sizeVariants');
        const row = document.createElement('div');
        row.className = 'variant-row';
        row.innerHTML = `
        <input type="text" name="size_name[]" placeholder="Size (e.g., M)">
        <input type="number" name="size_price[]" placeholder="Price adjustment (₹)" min="0" value="0">
        <button type="button" class="btn-remove-variant" onclick="removeVariant(this)">×</button>
    `;
        container.appendChild(row);
    }

    function removeVariant(btn) {
        const row = btn.closest('.variant-row');
        const container = row.parentElement;
        if (container.children.length > 1) {
            row.remove();
        } else {
            // Clear inputs instead of removing last row
            row.querySelectorAll('input').forEach(input => input.value = input.type === 'number' ? '0' : '');
        }
    }
</script>

<?php include 'includes/footer.php'; ?>