<?php

/**
 * Store Creation Wizard - Database Migration
 * Run this file once to set up wizard tables
 */
session_start();
require_once dirname(__DIR__, 2) . '/includes/auth.php';

$pdo = db();
$results = [];

try {
    // 1. Add category column to stores if not exists
    $stmt = $pdo->query("SHOW COLUMNS FROM stores LIKE 'category'");
    if ($stmt->rowCount() === 0) {
        $pdo->exec("ALTER TABLE stores ADD COLUMN category VARCHAR(50) NULL AFTER whatsapp");
        $results[] = "✓ Added 'category' column to stores table";
    } else {
        $results[] = "○ Column 'category' already exists";
    }

    // 2. Add city column to stores if not exists
    $stmt = $pdo->query("SHOW COLUMNS FROM stores LIKE 'city'");
    if ($stmt->rowCount() === 0) {
        $pdo->exec("ALTER TABLE stores ADD COLUMN city VARCHAR(100) NULL AFTER category");
        $results[] = "✓ Added 'city' column to stores table";
    } else {
        $results[] = "○ Column 'city' already exists";
    }

    // 3. Create business_categories table
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `business_categories` (
            `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
            `name` VARCHAR(50) NOT NULL,
            `icon` VARCHAR(10) NOT NULL,
            `display_order` INT NOT NULL DEFAULT 0,
            PRIMARY KEY (`id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    $results[] = "✓ Created 'business_categories' table";

    // 4. Seed categories if empty
    $stmt = $pdo->query("SELECT COUNT(*) FROM business_categories");
    if ($stmt->fetchColumn() == 0) {
        $pdo->exec("
            INSERT INTO `business_categories` (`name`, `icon`, `display_order`) VALUES
            ('Fashion & Clothing', '👗', 1),
            ('Electronics', '📱', 2),
            ('Home & Kitchen', '🏠', 3),
            ('Beauty & Health', '💄', 4),
            ('Food & Beverages', '🍕', 5),
            ('Jewellery', '💎', 6),
            ('Books & Stationery', '📚', 7),
            ('Sports & Fitness', '🏋️', 8),
            ('Toys & Games', '🎮', 9),
            ('Other', '📦', 10)
        ");
        $results[] = "✓ Seeded 10 business categories";
    } else {
        $results[] = "○ Business categories already seeded";
    }

    // 5. Create store_templates table
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `store_templates` (
            `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
            `name` VARCHAR(50) NOT NULL,
            `slug` VARCHAR(50) NOT NULL,
            `preview_color` VARCHAR(7) NOT NULL DEFAULT '#25D366',
            `theme_config` JSON NOT NULL,
            `is_active` TINYINT(1) NOT NULL DEFAULT 1,
            PRIMARY KEY (`id`),
            UNIQUE KEY `slug` (`slug`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    $results[] = "✓ Created 'store_templates' table";

    // 6. Seed templates if empty
    $stmt = $pdo->query("SELECT COUNT(*) FROM store_templates");
    if ($stmt->fetchColumn() == 0) {
        $pdo->exec("
            INSERT INTO `store_templates` (`name`, `slug`, `preview_color`, `theme_config`) VALUES
            ('Modern Minimal', 'modern-minimal', '#25D366', '{\"primary_color\":\"#25D366\",\"font\":\"Inter\",\"style\":\"minimal\"}'),
            ('Bold & Bright', 'bold-bright', '#6366f1', '{\"primary_color\":\"#6366f1\",\"font\":\"Poppins\",\"style\":\"bold\"}'),
            ('Classic Elegant', 'classic-elegant', '#0f172a', '{\"primary_color\":\"#0f172a\",\"font\":\"Playfair Display\",\"style\":\"elegant\"}'),
            ('Warm & Cozy', 'warm-cozy', '#f59e0b', '{\"primary_color\":\"#f59e0b\",\"font\":\"Nunito\",\"style\":\"warm\"}')
        ");
        $results[] = "✓ Seeded 4 store templates";
    } else {
        $results[] = "○ Store templates already seeded";
    }

    $success = true;
} catch (PDOException $e) {
    $results[] = "✗ Error: " . $e->getMessage();
    $success = false;
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Wizard Migration - ZITEO</title>
    <style>
        body {
            font-family: 'Inter', sans-serif;
            max-width: 600px;
            margin: 50px auto;
            padding: 20px;
        }

        h1 {
            color: #25D366;
        }

        .result {
            padding: 8px 12px;
            margin: 4px 0;
            border-radius: 6px;
        }

        .success {
            background: #f0fdf4;
            color: #166534;
        }

        .info {
            background: #f1f5f9;
            color: #475569;
        }

        .error {
            background: #fef2f2;
            color: #dc2626;
        }

        .back-link {
            display: inline-block;
            margin-top: 20px;
            color: #25D366;
        }
    </style>
</head>

<body>
    <h1>🛠️ Wizard Migration</h1>
    <?php foreach ($results as $result): ?>
        <div class="result <?= str_starts_with($result, '✓') ? 'success' : (str_starts_with($result, '○') ? 'info' : 'error') ?>">
            <?= htmlspecialchars($result) ?>
        </div>
    <?php endforeach; ?>

    <?php if ($success): ?>
        <p style="margin-top: 20px;">✅ Migration completed successfully!</p>
        <a href="wizard/wizard.php" class="back-link">→ Go to Store Creation Wizard</a>
    <?php endif; ?>
</body>

</html>