<?php

/**
 * Store Router
 * Routes all requests to appropriate template pages
 * URL Format: /{store-slug}/{page}/{params}
 */

require_once dirname(__DIR__, 2) . '/includes/db.php';
require_once dirname(__DIR__, 2) . '/includes/functions.php';

// Get store slug from subdomain or query param
$host = $_SERVER['HTTP_HOST'] ?? '';
$slug = '';

// Check if subdomain format: {slug}.my.ziteo.in
if (preg_match('/^([a-z0-9-]+)\.my\.ziteo\.in$/', $host, $matches)) {
    $slug = $matches[1];
    $storeUrl = '';
    $assetsUrl = ''; // Valid for subdomain root (maps to store/ via .htaccess)
    $uploadsUrl = ''; // Valid for subdomain root (excluded in .htaccess)
} else {
    // Fallback to query param for local development
    $slug = $_GET['store'] ?? '';
    $storeUrl = "/store?store=$slug";
    // For local dev, assets are in current folder relative to server root
    $scriptDir = dirname($_SERVER['SCRIPT_NAME']);
    $assetsUrl = rtrim($scriptDir, '/\\');
    $uploadsUrl = dirname($assetsUrl); // Parent directory for uploads
}

if (empty($slug)) {
    header('Location: https://ziteo.in');
    exit;
}

$pdo = db();

// Get store
$stmt = $pdo->prepare("
    SELECT s.*, u.name as owner_name
    FROM stores s
    JOIN users u ON s.user_id = u.id
    WHERE s.slug = ?
");
$stmt->execute([$slug]);
$store = $stmt->fetch();

if (!$store) {
    http_response_code(404);
    echo "<h1>Store not found</h1>";
    exit;
}

if ($store['status'] === 'pending') {
    echo "<h1>Coming Soon</h1><p>This store is being set up. Check back later!</p>";
    exit;
}

if ($store['status'] === 'suspended') {
    http_response_code(403);
    echo "<h1>Store Unavailable</h1>";
    exit;
}

// Theme configuration
$theme = json_decode($store['theme_config'] ?? '{}', true) ?: [];
$businessCategory = $theme['business_category'] ?? 'general';

// Parse the request path
$requestUri = $_SERVER['REQUEST_URI'] ?? '/';
$path = parse_url($requestUri, PHP_URL_PATH);

// Remove /store prefix if present (for local dev)
$path = preg_replace('#^/store#', '', $path);
$path = trim($path, '/');
$pathParts = $path ? explode('/', $path) : [];

// Determine current page
$page = $pathParts[0] ?? 'home';
$pageParam = $pathParts[1] ?? null;

// Fix store URL for local dev
if (strpos($requestUri, '/store?store=') !== false) {
    $storeUrl = "/store?store=$slug&page=";
    $page = $_GET['page'] ?? 'home';
    $pageParam = $_GET['id'] ?? null;
}

// Load template configuration
$templatePath = __DIR__ . '/templates/default';
$templateConfig = require $templatePath . '/config.php';

// Get categories for navigation/filters
try {
    $stmt = $pdo->prepare("
        SELECT pc.*, COUNT(p.id) as product_count
        FROM product_categories pc
        LEFT JOIN products p ON p.category_id = pc.id AND p.is_active = 1
        WHERE (pc.store_id = ? OR pc.store_id IS NULL) 
          AND pc.parent_category = ?
          AND pc.is_active = 1
        GROUP BY pc.id
        ORDER BY pc.name
    ");
    $stmt->execute([$store['id'], $businessCategory]);
    $categories = $stmt->fetchAll();
} catch (PDOException $e) {
    $categories = [];
}

// Route to appropriate page
switch ($page) {
    case '':
    case 'home':
        $currentPage = 'home';
        $pageTitle = $store['name'] . ' - Home';

        // Get featured products
        $stmt = $pdo->prepare("
            SELECT p.*, pc.name as category_name 
            FROM products p 
            LEFT JOIN product_categories pc ON p.category_id = pc.id
            WHERE p.store_id = ? AND p.is_active = 1 
            ORDER BY p.created_at DESC 
            LIMIT 8
        ");
        $stmt->execute([$store['id']]);
        $featuredProducts = $stmt->fetchAll();

        include $templatePath . '/header.php';
        include $templatePath . '/home.php';
        include $templatePath . '/footer.php';
        break;

    case 'products':
        $currentPage = 'products';
        $pageTitle = 'All Products - ' . $store['name'];

        // Build query with filters
        $sql = "SELECT p.*, pc.name as category_name 
                FROM products p 
                LEFT JOIN product_categories pc ON p.category_id = pc.id
                WHERE p.store_id = ? AND p.is_active = 1";
        $params = [$store['id']];

        // Category filter
        if (!empty($_GET['category'])) {
            $sql .= " AND p.category_id = ?";
            $params[] = (int)$_GET['category'];
        }

        // Price filter
        if (!empty($_GET['min_price'])) {
            $sql .= " AND (p.offer_price >= ? OR (p.offer_price IS NULL AND p.price >= ?))";
            $minPricePaise = (int)$_GET['min_price'] * 100;
            $params[] = $minPricePaise;
            $params[] = $minPricePaise;
        }
        if (!empty($_GET['max_price'])) {
            $sql .= " AND (p.offer_price <= ? OR (p.offer_price IS NULL AND p.price <= ?))";
            $maxPricePaise = (int)$_GET['max_price'] * 100;
            $params[] = $maxPricePaise;
            $params[] = $maxPricePaise;
        }

        // Sort
        $sort = $_GET['sort'] ?? 'newest';
        switch ($sort) {
            case 'price_low':
                $sql .= " ORDER BY COALESCE(p.offer_price, p.price) ASC";
                break;
            case 'price_high':
                $sql .= " ORDER BY COALESCE(p.offer_price, p.price) DESC";
                break;
            case 'name_asc':
                $sql .= " ORDER BY p.name ASC";
                break;
            default:
                $sql .= " ORDER BY p.created_at DESC";
        }

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $products = $stmt->fetchAll();

        include $templatePath . '/header.php';
        include $templatePath . '/products.php';
        include $templatePath . '/footer.php';
        break;

    case 'product':
        $currentPage = 'product';
        $productId = (int)$pageParam;

        if (!$productId) {
            header("Location: $storeUrl/products");
            exit;
        }

        $stmt = $pdo->prepare("
            SELECT p.*, pc.name as category_name 
            FROM products p 
            LEFT JOIN product_categories pc ON p.category_id = pc.id
            WHERE p.id = ? AND p.store_id = ?
        ");
        $stmt->execute([$productId, $store['id']]);
        $product = $stmt->fetch();

        if (!$product) {
            http_response_code(404);
            echo "<h1>Product not found</h1>";
            exit;
        }

        $pageTitle = $product['name'] . ' - ' . $store['name'];
        $pageDescription = substr(strip_tags($product['description'] ?? ''), 0, 160);

        include $templatePath . '/header.php';
        include $templatePath . '/product.php';
        include $templatePath . '/footer.php';
        break;

    case 'search':
        $currentPage = 'search';
        $pageTitle = 'Search - ' . $store['name'];
        $query = trim($_GET['q'] ?? '');

        $products = [];
        if (!empty($query)) {
            $stmt = $pdo->prepare("
                SELECT p.*, pc.name as category_name 
                FROM products p 
                LEFT JOIN product_categories pc ON p.category_id = pc.id
                WHERE p.store_id = ? AND p.is_active = 1 
                  AND (p.name LIKE ? OR p.description LIKE ?)
                ORDER BY p.created_at DESC
            ");
            $searchTerm = "%$query%";
            $stmt->execute([$store['id'], $searchTerm, $searchTerm]);
            $products = $stmt->fetchAll();
        }

        // Get recent products for suggestions
        $stmt = $pdo->prepare("
            SELECT p.*, pc.name as category_name 
            FROM products p 
            LEFT JOIN product_categories pc ON p.category_id = pc.id
            WHERE p.store_id = ? AND p.is_active = 1 
            ORDER BY p.created_at DESC 
            LIMIT 4
        ");
        $stmt->execute([$store['id']]);
        $recentProducts = $stmt->fetchAll();

        include $templatePath . '/header.php';
        include $templatePath . '/search.php';
        include $templatePath . '/footer.php';
        break;

    case 'about':
        $currentPage = 'about';
        $pageTitle = 'About - ' . $store['name'];

        include $templatePath . '/header.php';
        include $templatePath . '/about.php';
        include $templatePath . '/footer.php';
        break;

    case 'wishlist':
        $currentPage = 'wishlist';
        $pageTitle = 'Wishlist - ' . $store['name'];

        include $templatePath . '/header.php';
        include $templatePath . '/wishlist.php';
        include $templatePath . '/footer.php';
        break;

    default:
        // 404 page
        http_response_code(404);
        $currentPage = '404';
        $pageTitle = 'Page Not Found';
        include $templatePath . '/header.php';
        echo '<div class="container"><div class="empty-state"><h1>404</h1><p>Page not found</p><a href="' . $storeUrl . '/" class="btn btn-primary">Go Home</a></div></div>';
        include $templatePath . '/footer.php';
}
