<?php

/**
 * Wishlist Page Template
 * Products stored in localStorage
 */
?>

<div class="container">
    <div class="wishlist-page">
        <div class="wishlist-header">
            <h1>❤️ My Wishlist</h1>
            <p class="wishlist-subtitle">Your saved products</p>
        </div>

        <div class="wishlist-content" id="wishlistContent">
            <!-- Loading state -->
            <div class="wishlist-loading" id="wishlistLoading">
                <p>Loading your wishlist...</p>
            </div>

            <!-- Empty state (shown via JS) -->
            <div class="empty-state wishlist-empty" id="wishlistEmpty" style="display: none;">
                <span class="empty-icon">💔</span>
                <h3>Your wishlist is empty</h3>
                <p>Save products you love by clicking the heart icon</p>
                <a href="<?= $storeUrl ?>/products" class="btn btn-primary">Browse Products</a>
            </div>

            <!-- Wishlist items (populated via JS) -->
            <div class="wishlist-grid" id="wishlistGrid"></div>
        </div>

        <!-- Actions -->
        <div class="wishlist-actions" id="wishlistActions" style="display: none;">
            <button class="btn btn-outline" onclick="clearWishlist()">
                🗑️ Clear Wishlist
            </button>
            <a href="#" id="orderAllBtn" class="btn btn-whatsapp" target="_blank">
                💬 Order All on WhatsApp
            </a>
        </div>
    </div>
</div>

<script>
    const storeUrl = '<?= $storeUrl ?>';
    const whatsappNumber = '91<?= e($store['whatsapp']) ?>';

    function renderWishlist() {
        const wishlist = JSON.parse(localStorage.getItem('wishlist_' + STORE_ID) || '[]');
        const grid = document.getElementById('wishlistGrid');
        const empty = document.getElementById('wishlistEmpty');
        const loading = document.getElementById('wishlistLoading');
        const actions = document.getElementById('wishlistActions');

        loading.style.display = 'none';

        if (wishlist.length === 0) {
            empty.style.display = 'block';
            grid.style.display = 'none';
            actions.style.display = 'none';
            return;
        }

        empty.style.display = 'none';
        grid.style.display = 'grid';
        actions.style.display = 'flex';

        grid.innerHTML = wishlist.map(item => `
        <div class="wishlist-card" data-product-id="${item.id}">
            <a href="${storeUrl}/product/${item.id}" class="wishlist-image">
                ${item.image 
                    ? `<img src="${item.image}" alt="${escapeHtml(item.name)}" loading="lazy">`
                    : '<div class="no-image">📦</div>'}
            </a>
            <div class="wishlist-info">
                <h3><a href="${storeUrl}/product/${item.id}">${escapeHtml(item.name)}</a></h3>
                <p class="wishlist-price">${formatPrice(item.price)}</p>
                <div class="wishlist-item-actions">
                    <button class="btn btn-outline btn-sm" onclick="removeAndRender(${item.id})">
                        Remove
                    </button>
                    <a href="${storeUrl}/product/${item.id}" class="btn btn-primary btn-sm">
                        View
                    </a>
                </div>
            </div>
        </div>
    `).join('');

        // Update order all link
        updateOrderAllLink(wishlist);
    }

    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    function formatPrice(paise) {
        return '₹' + (paise / 100).toLocaleString('en-IN');
    }

    function removeAndRender(productId) {
        removeFromWishlist(productId);
        renderWishlist();
        showToast('Removed from wishlist');
    }

    function clearWishlist() {
        if (confirm('Are you sure you want to clear your wishlist?')) {
            localStorage.removeItem('wishlist_' + STORE_ID);
            renderWishlist();
            updateWishlistCount();
            showToast('Wishlist cleared');
        }
    }

    function updateOrderAllLink(wishlist) {
        if (wishlist.length === 0) return;

        let message = 'Hi, I want to order the following items:\n\n';
        let total = 0;

        wishlist.forEach((item, index) => {
            message += `${index + 1}. ${item.name} - ${formatPrice(item.price)}\n`;
            total += item.price;
        });

        message += `\nTotal: ${formatPrice(total)}`;

        document.getElementById('orderAllBtn').href =
            `https://wa.me/${whatsappNumber}?text=${encodeURIComponent(message)}`;
    }

    // Initialize
    document.addEventListener('DOMContentLoaded', renderWishlist);
</script>