<?php

/**
 * Store Owner Dashboard - Main Dashboard
 * app.ziteo.in/dashboard.php
 */
session_start();
require_once dirname(__DIR__, 2) . '/includes/auth.php';
requireAuth();

$user = getCurrentUser();

// If user not found in database, force logout
if (!$user) {
    logoutUser();
    header('Location: login.php');
    exit;
}

$pdo = db();

// Check if a specific store is selected
$selectedStoreId = getSelectedStoreId();
$selectedStore = null;
if ($selectedStoreId) {
    $selectedStore = validateStoreOwnership($selectedStoreId);
    if (!$selectedStore) {
        // Store doesn't exist or user doesn't own it
        header('Location: dashboard.php');
        exit;
    }
}

// Get Plan Name
try {
    $stmt = $pdo->prepare("SELECT p.name FROM plans p JOIN users u ON u.plan_id = p.id WHERE u.id = ?");
    $stmt->execute([$user['id']]);
    $user['plan_name'] = $stmt->fetchColumn();
} catch (PDOException $e) {
    $user['plan_name'] = 'Fix Database';
    $db_error = true;
}

// Get plan limits
try {
    $stmt = $pdo->prepare("SELECT store_limit FROM plans WHERE id = ?");
    $stmt->execute([$user['plan_id']]);
    $storeLimit = $stmt->fetchColumn() ?: 1;
} catch (PDOException $e) {
    $storeLimit = 1;
}

// Get ALL user's stores
$stmt = $pdo->prepare("SELECT * FROM stores WHERE user_id = ? ORDER BY created_at DESC");
$stmt->execute([currentUserId()]);
$stores = $stmt->fetchAll();
$storeCount = count($stores);
$canCreateStore = $storeCount < $storeLimit;

// If viewing a specific store, get its stats
$stats = ['products' => 0, 'orders' => 0, 'orders_week' => 0];
if ($selectedStore) {
    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM products WHERE store_id = ?");
    $stmt->execute([$selectedStore['id']]);
    $stats['products'] = $stmt->fetch()['count'];

    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM orders WHERE store_id = ?");
    $stmt->execute([$selectedStore['id']]);
    $stats['orders'] = $stmt->fetch()['count'];

    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM orders WHERE store_id = ? AND created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)");
    $stmt->execute([$selectedStore['id']]);
    $stats['orders_week'] = $stmt->fetch()['count'];
}

// Get stats for each store (for store cards)
$storeStats = [];
foreach ($stores as $store) {
    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM products WHERE store_id = ?");
    $stmt->execute([$store['id']]);
    $productCount = $stmt->fetch()['count'];

    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM orders WHERE store_id = ?");
    $stmt->execute([$store['id']]);
    $orderCount = $stmt->fetch()['count'];

    $storeStats[$store['id']] = [
        'products' => $productCount,
        'orders' => $orderCount
    ];
}

include 'includes/header.php';
?>

<div class="dashboard-content">
    <?php if (isset($db_error)): ?>
        <div class="bg-red-50 border-l-4 border-red-400 p-4 mb-6">
            <div class="flex">
                <div class="flex-shrink-0">⚠️</div>
                <div class="ml-3">
                    <p class="text-sm text-red-700">
                        Database needs an update. <a href="migrate.php" class="font-bold underline">Click here to fix it now</a>.
                    </p>
                </div>
            </div>
        </div>
    <?php endif; ?>

    <?php if (empty($user['plan_id'])): ?>
        <script>
            window.location.href = 'select_plan.php';
        </script>
    <?php elseif ($user['plan_status'] !== 'active'): ?>
        <!-- Plan Pending -->
        <div class="empty-state">
            <div class="empty-state-icon">⏳</div>
            <h2>Plan Pending Approval</h2>
            <p>You have selected the <strong><?= e($user['plan_name'] ?? 'Selected') ?></strong> plan.</p>
            <p>Please wait while we verify your account. Contact support for faster approval.</p>
        </div>
    <?php elseif ($selectedStore): ?>
        <!-- Store-specific Dashboard View -->
        <div class="dashboard-header">
            <div>
                <h1><?= e($selectedStore['name']) ?></h1>
                <p class="store-link">
                    <a href="https://<?= e($selectedStore['slug']) ?>.my.ziteo.in" target="_blank">
                        <?= e($selectedStore['slug']) ?>.my.ziteo.in
                    </a>
                    <button onclick="copyLink('<?= e($selectedStore['slug']) ?>')" class="btn-copy" title="Copy link">📋</button>
                </p>
            </div>
            <?php if ($selectedStore['status'] === 'pending'): ?>
                <span class="status-badge status-pending">Under Review</span>
            <?php else: ?>
                <span class="status-badge status-active">Active</span>
            <?php endif; ?>
        </div>

        <?php if ($selectedStore['status'] === 'pending'): ?>
            <div class="empty-state">
                <div class="empty-state-icon">⏳</div>
                <h2>Store Under Review</h2>
                <p>Your store is being reviewed. This usually takes 24-48 hours.</p>
            </div>
        <?php else: ?>
            <!-- Stats -->
            <div class="stats-grid">
                <div class="stat-card">
                    <span class="stat-value"><?= $stats['products'] ?></span>
                    <span class="stat-label">Products</span>
                </div>
                <div class="stat-card">
                    <span class="stat-value"><?= $stats['orders_week'] ?></span>
                    <span class="stat-label">Orders This Week</span>
                </div>
                <div class="stat-card">
                    <span class="stat-value"><?= $stats['orders'] ?></span>
                    <span class="stat-label">Total Orders</span>
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="quick-actions">
                <h3>Quick Actions</h3>
                <div class="action-grid">
                    <a href="products.php?store=<?= $selectedStore['id'] ?>&action=add" class="action-card">
                        <span class="action-icon">➕</span>
                        <span>Add Product</span>
                    </a>
                    <a href="products.php?store=<?= $selectedStore['id'] ?>" class="action-card">
                        <span class="action-icon">📦</span>
                        <span>Manage Products</span>
                    </a>
                    <a href="orders.php?store=<?= $selectedStore['id'] ?>" class="action-card">
                        <span class="action-icon">📋</span>
                        <span>View Orders</span>
                    </a>
                    <a href="settings.php?store=<?= $selectedStore['id'] ?>" class="action-card">
                        <span class="action-icon">⚙️</span>
                        <span>Store Settings</span>
                    </a>
                    <a href="product-settings.php?store=<?= $selectedStore['id'] ?>" class="action-card">
                        <span class="action-icon">🎛️</span>
                        <span>Product Settings</span>
                    </a>
                </div>
            </div>
        <?php endif; ?>

    <?php else: ?>
        <!-- Stores List View -->
        <div class="dashboard-header">
            <div>
                <h1>My Stores</h1>
                <p class="subtitle">Manage your online stores</p>
            </div>
            <?php if ($canCreateStore): ?>
                <a href="create-store.php" class="btn btn-primary">+ Create New Store</a>
            <?php else: ?>
                <div class="limit-notice">
                    Store Limit Reached (<?= $storeCount ?>/<?= $storeLimit ?>).
                    <a href="select_plan.php">Upgrade Plan</a>
                </div>
            <?php endif; ?>
        </div>

        <?php if (empty($stores)): ?>
            <!-- No Stores - Empty State -->
            <div class="empty-state">
                <div class="empty-state-icon">🏪</div>
                <h2>Welcome, <?= e($user['name']) ?>!</h2>
                <p>You are on the <strong><?= e($user['plan_name'] ?? 'Basic') ?></strong> plan.</p>
                <p>Let's create your first store and start selling.</p>
                <a href="create-store.php" class="btn btn-primary btn-large">Create Your Store →</a>
            </div>
        <?php else: ?>
            <!-- Stores Grid -->
            <div class="stores-grid">
                <?php foreach ($stores as $store): ?>
                    <a href="dashboard.php?store=<?= $store['id'] ?>" class="store-card">
                        <div class="store-card-header">
                            <h3><?= e($store['name']) ?></h3>
                            <span class="status-badge status-<?= $store['status'] ?>">
                                <?= ucfirst($store['status']) ?>
                            </span>
                        </div>
                        <p class="store-url"><?= e($store['slug']) ?>.my.ziteo.in</p>
                        <div class="store-stats">
                            <div class="store-stat">
                                <span class="store-stat-value"><?= $storeStats[$store['id']]['products'] ?></span>
                                <span class="store-stat-label">Products</span>
                            </div>
                            <div class="store-stat">
                                <span class="store-stat-value"><?= $storeStats[$store['id']]['orders'] ?></span>
                                <span class="store-stat-label">Orders</span>
                            </div>
                        </div>
                        <div class="store-card-action">
                            Manage Store →
                        </div>
                    </a>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    <?php endif; ?>
</div>

<script>
    function copyLink(slug) {
        navigator.clipboard.writeText('https://' + slug + '.my.ziteo.in');
        alert('Link copied!');
    }
</script>

<?php include 'includes/footer.php'; ?>