<?php

/**
 * Orders Management Page
 */
session_start();
require_once dirname(__DIR__, 2) . '/includes/auth.php';
requireAuth();

// Get store from URL parameter
$storeId = getSelectedStoreId();
if (!$storeId) {
    header('Location: dashboard.php');
    exit;
}

// Validate store ownership
$store = validateStoreOwnership($storeId);
if (!$store) {
    header('Location: dashboard.php');
    exit;
}

$pdo = db();

// Handle status update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['order_id'], $_POST['status'])) {
    $orderId = (int)$_POST['order_id'];
    $status = $_POST['status'];

    $validStatuses = ['initiated', 'confirmed', 'shipped', 'delivered', 'cancelled'];
    if (in_array($status, $validStatuses)) {
        $stmt = $pdo->prepare("UPDATE orders SET status = ? WHERE id = ? AND store_id = ?");
        $stmt->execute([$status, $orderId, $storeId]);
    }

    header("Location: orders.php?store=$storeId");
    exit;
}

// Get orders
$stmt = $pdo->prepare("SELECT * FROM orders WHERE store_id = ? ORDER BY created_at DESC");
$stmt->execute([$storeId]);
$orders = $stmt->fetchAll();

include 'includes/header.php';
?>

<div class="dashboard-content">
    <div class="page-header">
        <h1>Orders</h1>
    </div>

    <?php if (empty($orders)): ?>
        <div class="empty-state-small">
            <p>No orders yet. Share your store link to start receiving orders!</p>
        </div>
    <?php else: ?>
        <div class="orders-table-wrapper">
            <table class="orders-table">
                <thead>
                    <tr>
                        <th>Order #</th>
                        <th>Items</th>
                        <th>Total</th>
                        <th>Status</th>
                        <th>Date</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($orders as $order): ?>
                        <?php $items = json_decode($order['items'], true); ?>
                        <tr>
                            <td><strong><?= e($order['order_number']) ?></strong></td>
                            <td>
                                <?php foreach ($items as $item): ?>
                                    <div><?= e($item['name']) ?> × <?= $item['quantity'] ?></div>
                                <?php endforeach; ?>
                            </td>
                            <td><?= formatPrice($order['total']) ?></td>
                            <td>
                                <span class="status-badge status-<?= $order['status'] ?>">
                                    <?= ucfirst($order['status']) ?>
                                </span>
                            </td>
                            <td><?= date('d M Y, h:i A', strtotime($order['created_at'])) ?></td>
                            <td>
                                <form method="POST" class="status-form">
                                    <input type="hidden" name="order_id" value="<?= $order['id'] ?>">
                                    <select name="status" onchange="this.form.submit()">
                                        <option value="initiated" <?= $order['status'] === 'initiated' ? 'selected' : '' ?>>Initiated</option>
                                        <option value="confirmed" <?= $order['status'] === 'confirmed' ? 'selected' : '' ?>>Confirmed</option>
                                        <option value="shipped" <?= $order['status'] === 'shipped' ? 'selected' : '' ?>>Shipped</option>
                                        <option value="delivered" <?= $order['status'] === 'delivered' ? 'selected' : '' ?>>Delivered</option>
                                        <option value="cancelled" <?= $order['status'] === 'cancelled' ? 'selected' : '' ?>>Cancelled</option>
                                    </select>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>

<?php include 'includes/footer.php'; ?>